<?php

namespace App\Controller\Domain\backend;

use App\Entity\Partner;
use App\Form\PartnerType;
use App\Helpers\Utility;
use App\Repository\PartnerRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/admin/partner', name: 'back.partner.')]
final class PartnerController extends AbstractController
{
    private string $basePath = "uploads/images/partners/";

    #[Route(name: 'index', methods: ['GET'])]
    public function index(PartnerRepository $partnerRepository): Response
    {
        return $this->render('domain/backend/partner/index.html.twig', [
            'partners' => $partnerRepository->findAll(),
        ]);
    }

    #[Route('/new', name: 'new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $partner = new Partner();
        $form = $this->createForm(PartnerType::class, $partner);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $imageFile = $form->get('image')->getData();
            $fileStored = Utility::uploadFile($imageFile, $this->getParameter('images_directory'),"partners");
            if ($fileStored) {
                $partner->setImage($this->basePath .$fileStored);
            }
            $entityManager->persist($partner);
            $entityManager->flush();

            return $this->redirectToRoute('back.partner.index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('domain/backend/partner/new.html.twig', [
            'partner' => $partner,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'show', methods: ['GET'])]
    public function show(Partner $partner): Response
    {
        return $this->render('domain/backend/partner/show.html.twig', [
            'partner' => $partner,
        ]);
    }

    #[Route('/{id}/edit', name: 'edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Partner $partner, EntityManagerInterface $entityManager): Response
    {
        $oldImage = $partner->getImage();
        $form = $this->createForm(PartnerType::class, $partner);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $imageFile = $form->get('image')->getData();
            $fileStored = Utility::uploadFile($imageFile, $this->getParameter('images_directory'),"partners");
            if ($fileStored) {
                // Suppression de l'ancienne image
                $path = $this->getParameter('images_directory') . '/partners/' . basename($oldImage);
                if ($oldImage && file_exists($path)) {
                    unlink($path);
                }
                $partner->setImage($this->basePath .$fileStored);
            }
            $entityManager->flush();

            return $this->redirectToRoute('back.partner.index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('domain/backend/partner/edit.html.twig', [
            'partner' => $partner,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'delete', methods: ['POST'])]
    public function delete(Request $request, Partner $partner, EntityManagerInterface $entityManager): Response
    {
        if ($this->isCsrfTokenValid('delete'.$partner->getId(), $request->getPayload()->getString('_token'))) {
            $image = $partner->getImage();
            if ($image) {
                $path = $this->getParameter('images_directory') . '/partners/' . basename($image);
                if (file_exists($path)) {
                    unlink($path);
                }
            }
            $entityManager->remove($partner);
            $entityManager->flush();
        }

        return $this->redirectToRoute('back.partner.index', [], Response::HTTP_SEE_OTHER);
    }
}
