<?php

namespace App\Controller\Domain\backend;

use App\Entity\Post;
use App\Entity\PostMedia;
use App\Form\PostType;
use App\Helpers\Utility;
use App\Repository\PostRepository;
use App\Service\SlugService;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\File\Exception\FileException;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/admin/post', name: 'back.post.')]
final class PostController extends AbstractController
{
    private string $basePath = "uploads/images/posts/";
    private string $MediaBasePath = "uploads/images/medias/";

    #[Route(name: 'index', methods: ['GET'])]
    public function index(PostRepository $postRepository): Response
    {
        return $this->render('domain/backend/post/index.html.twig', [
            'posts' => $postRepository->findAll(),
        ]);
    }

    #[Route('/new', name: 'new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $post = new Post();
        $form = $this->createForm(PostType::class, $post);
        $form->handleRequest($request);
        $files = [];

        if ($form->isSubmitted() && $form->isValid()) {
            try {
                $imageFile = $form->get('imageCouverture')->getData();
                $fileStored = Utility::uploadFile($imageFile, $this->getParameter('images_directory'), "posts", 1024, 683);
                if ($fileStored) {
                    $post->setImageCouverture($this->basePath . $fileStored);
                    $files[] = $this->getParameter('images_directory')."/posts/". basename($post->getImageCouverture());
                }

                $postMedias = $form->get('postMedia')->getData();
                foreach ($postMedias as $postMedia) {
                    if ($postMedia instanceof PostMedia) {
                        $mediaFile = $postMedia->getFile();

                        $newFilename = Utility::uploadFile($mediaFile, $this->getParameter('images_directory'), "medias", 1024, 683);
                        if ($newFilename) {
                            $postMedia->setMedia($this->MediaBasePath . $newFilename);
                            $files[] = $this->getParameter('images_directory')."/medias/". basename($newFilename);
                        }

                        $post->addPostMedia($postMedia);
                        $postMedia->setPost($post);
                        $entityManager->persist($postMedia);
                    }
                }

                $entityManager->persist($post);
                $entityManager->flush();
            }catch (\Throwable  $exception){
                foreach ($files as $file) {
                    if (file_exists($file)) {
                        unlink($file);
                    }
                }
                throw $exception;
            }

            return $this->redirectToRoute('back.post.index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('domain/backend/post/new.html.twig', [
            'post' => $post,
            'form' => $form,
        ]);
    }

    #[Route('/{slug}', name: 'show', methods: ['GET'])]
    public function show(Post $post): Response
    {
        return $this->render('domain/backend/post/show.html.twig', [
            'post' => $post,
        ]);
    }

    #[Route('/{slug}/edit', name: 'edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, $slug, EntityManagerInterface $entityManager, PostRepository $postRepository): Response
    {
        $post = $postRepository->findOneBy(['slug' => $slug]);
        // Sauvegardons les anciens chemins
        $oldCoverImage = $post->getImageCouverture();
        $oldMedias = $post->getPostMedia()->toArray();

        $form = $this->createForm(PostType::class, $post);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Gestion de l'image de couverture
            $imageFile = $form->get('imageCouverture')->getData();
            if ($imageFile) {
                $fileStored = Utility::uploadFile($imageFile, $this->getParameter('images_directory'), "posts", 1024, 683);
                if ($fileStored) {
                    // Suppression de l'ancienne image de couverture
                    $path = $this->getParameter('images_directory') . '/posts/' . basename($oldCoverImage);
                    if ($oldCoverImage && file_exists($path)) {
                        unlink($path);
                    }
                    $post->setImageCouverture($this->basePath . $fileStored);
                }
            }

            // Gestion des médias
            $newMedias = $form->get('postMedia')->getData();
            $currentMediaPaths = []; // Pour tracker les médias conservés

            foreach ($newMedias as $postMedia) {
                if ($postMedia instanceof PostMedia) {
                    /** @var UploadedFile $mediaFile */
                    $mediaFile = $postMedia->getFile();
                    if ($mediaFile) {
                        $newFilename = Utility::uploadFile($mediaFile, $this->getParameter('images_directory'), "medias", 1024, 683);
                        if ($newFilename) {
                            $postMedia->setMedia($this->MediaBasePath . $newFilename);
                        }
                    }
                    $currentMediaPaths[] = $postMedia->getMedia();
                    $post->addPostMedia($postMedia);
                    $postMedia->setPost($post);
                    $entityManager->persist($postMedia);
                }
            }

            // Suppression des anciens médias qui ne sont plus utilisés
            foreach ($oldMedias as $oldMedia) {
                if (!in_array($oldMedia->getMedia(), $currentMediaPaths)) {
                    $oldFilePath = $this->getParameter('images_directory') . '/medias/' . basename($oldMedia->getMedia());
                    if (file_exists($oldFilePath)) {
                        unlink($oldFilePath);
                    }
                    $entityManager->remove($oldMedia);
                }
            }

            $entityManager->persist($post);
            $entityManager->flush();

            return $this->redirectToRoute('back.post.index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('domain/backend/post/edit.html.twig', [
            'post' => $slug,
            'form' => $form,
        ]);
    }

    #[Route('/{slug}', name: 'delete', methods: ['POST'])]
    public function delete(Request $request, $slug, EntityManagerInterface $entityManager, PostRepository $postRepository): Response
    {
        $post = $postRepository->findOneBy(['slug' => $slug]);
        if ($this->isCsrfTokenValid('delete' . $post->getSlug(), $request->getPayload()->getString('_token'))) {
            // Suppression de l'image de couverture
            $coverImage = $post->getImageCouverture();
            if ($coverImage) {
                $path = $this->getParameter('images_directory') . '/posts/' . basename($coverImage);
                if (file_exists($path)) {
                    unlink($path);
                }
            }

            // Suppression des médias associés
            foreach ($post->getPostMedia() as $media) {
                $mediaPath = $this->getParameter('images_directory') . '/medias/' . basename($media->getMedia());
                if (file_exists($mediaPath)) {
                    unlink($mediaPath);
                }
            }
            $entityManager->remove($post);
            $entityManager->flush();
        }

        return $this->redirectToRoute('back.post.index', [], Response::HTTP_SEE_OTHER);
    }
}
