<?php

namespace App\Controller\Domain\backend;

use App\Entity\Slide;
use App\Form\SlideType;
use App\Helpers\Utility;
use App\Repository\SlideRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/admin/slide', name: 'back.slide.')]
final class SlideController extends AbstractController
{
    private string $basePath = "uploads/images/slides/";

    #[Route(name: 'index', methods: ['GET'])]
    public function index(SlideRepository $slideRepository): Response
    {
        return $this->render('domain/backend/slide/index.html.twig', [
            'slides' => $slideRepository->findAll(),
        ]);
    }

    #[Route('/new', name: 'new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $slide = new Slide();
        $form = $this->createForm(SlideType::class, $slide);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $imageFile = $form->get('image')->getData();
            $fileStored = Utility::uploadFile($imageFile, $this->getParameter('images_directory'), "slides", 1780, 830);
            if ($fileStored) {
                $slide->setImage($this->basePath . $fileStored);
            }
            $entityManager->persist($slide);
            $entityManager->flush();

            return $this->redirectToRoute('back.slide.index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('domain/backend/slide/new.html.twig', [
            'slide' => $slide,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'show', methods: ['GET'])]
    public function show(Slide $slide): Response
    {
        return $this->render('domain/backend/slide/show.html.twig', [
            'slide' => $slide,
        ]);
    }

    #[Route('/{id}/edit', name: 'edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Slide $slide, EntityManagerInterface $entityManager): Response
    {
        $oldImage = $slide->getImage();
        $form = $this->createForm(SlideType::class, $slide);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $imageFile = $form->get('image')->getData();
            $fileStored = Utility::uploadFile($imageFile, $this->getParameter('images_directory'), "slides", 1780, 830);
            if ($fileStored) {
                $slide->setImage($this->basePath . $fileStored);

                $path = $this->getParameter('images_directory') . '/slides/' . basename($fileStored);
                if ($oldImage && file_exists($path)) {
                    unlink($path);
                }
            }
            $entityManager->flush();

            return $this->redirectToRoute('back.slide.index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('domain/backend/slide/edit.html.twig', [
            'slide' => $slide,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'delete', methods: ['POST'])]
    public function delete(Request $request, Slide $slide, EntityManagerInterface $entityManager): Response
    {
        if ($this->isCsrfTokenValid('delete' . $slide->getId(), $request->getPayload()->getString('_token'))) {
            $image = $slide->getImage();
            $entityManager->remove($slide);
            $entityManager->flush();
            if ($image) {
                $path = $this->getParameter('images_directory') . '/slides/' . basename($image);
                if (file_exists($path)) {
                    unlink($path);
                }
            }
        }

        return $this->redirectToRoute('back.slide.index', [], Response::HTTP_SEE_OTHER);
    }
}
